/*
Copyright  2005, Apple Computer, Inc.  All rights reserved.
NOTE:  Use of this source code is subject to the terms of the Software
License Agreement for Mac OS X, which accompanies the code.  Your use
of this source code signifies your agreement to such license terms and
conditions.  Except as expressly granted in the Software License Agreement
for Mac OS X, no other copyright, patent, or other intellectual property
license or right is granted, either expressly or by implication, by Apple.
*/



//
// Terminology:
//
// The "expanded view" state is the larger (default) size, with the calendar day grid array (drawGrid) showing.
// The "collapsed view" state is the smaller size, with only the day-of-week and day-of-month-number showing.
//
// A "date id" (see idFromMonthDate()) is an html ID of the form "0102" (mmdd) that is used to refer to the
// dynamically-generated selectable calendar days in the grid.
//



var kWidgetCollapsedWidth = 167;
var kWidgetExpandedWidth = 333;
var kWidgetHeight = 163;

var kExpandedViewMiddleContainerWidth = 324;
var kExpandedViewMiddleFillWidth = 311;
var kExpandedViewRightContentLeftPosition = 179;

var kCollapsedViewMiddleContainerWidth = 155;
var kCollapsedViewMiddleFillWidth = 142;
var kCollapsedViewRightContentLeftPosition = 183;

var gResizeAnimation = {startTime:0, duration:250, positionFrom:0, positionTo:0, positionNow:0, timer:null, elementMiddleContainer:null, elementMiddleFill:null, elementRightContent:null, onfinished:null};
var gDate = new Date ();
var gBrowsingMonth = gDate.getMonth ();
var gBrowsingYear = gDate.getFullYear ();
var gBrowsingDate = gDate.getDate ();
var gIsCollapsed = false;
var gUpdateDateTimer = null;
var gButtonBeingTracked = null;



function keyPressed(e)
{
	if (!gIsCollapsed)
	{
		if (!gBrowsingDate)
			gBrowsingDate = 1;
		
		var mo = new Date(gBrowsingYear, gBrowsingMonth, gBrowsingDate);
		var dateChanged = false;
		
		switch (e.keyIdentifier) {
			case "Up":
				gBrowsingYear--;
				if (gBrowsingYear < 1753)			// assuming current calendar is Gregorian (not necessarily true), this
					gBrowsingYear = 1753;			// prevents incorrect viewing of September 1752 and before, including B.C.
				mo.setFullYear(gBrowsingYear);
				dateChanged = true;
			break;
			
			case "Down":
				gBrowsingYear++;
				mo.setFullYear(gBrowsingYear);
				dateChanged = true;
			break;
			
			case "Left":
			case "PageUp":
				gBrowsingMonth--;
				if (gBrowsingMonth < 0) {
					gBrowsingMonth = 11;
					gBrowsingYear--;
					if (gBrowsingYear < 1753)		// pin backwards monthwise browsing at January 1753; see note above
					{
						gBrowsingMonth = 0;
						gBrowsingYear = 1753;
					}
				}
				mo.setMonth(gBrowsingMonth);
				mo.setFullYear(gBrowsingYear);
				dateChanged = true;
			break;
			
			case "Right":
			case "PageDown":
				gBrowsingMonth++;
				if (gBrowsingMonth > 11) {
					gBrowsingMonth = 0;
					gBrowsingYear++;
				}
				mo.setMonth(gBrowsingMonth);
				mo.setFullYear(gBrowsingYear);
				dateChanged = true;
			break;
			
			case "Home":
				var today = new Date();
				gBrowsingMonth = today.getMonth();
				mo.setMonth(gBrowsingMonth);
				gBrowsingYear = today.getFullYear();
				mo.setFullYear(gBrowsingYear);
				gBrowsingDate = today.getDate();
				mo.setDate(gBrowsingDate);
				dateChanged = true;
				break;
			
			default:
			break;
		}
	
		if (dateChanged)
		{
			// create the new calendar
			gBrowsingMonth = mo.getMonth();
			gBrowsingYear = mo.getFullYear();
			gBrowsingDate = mo.getDate();
			drawGrid(gBrowsingMonth, gBrowsingYear);
			
			drawHiliteTodayInGrid ();
			
			gBrowsingDate = 0; // note this makes the id of the form mm00yyyy which doesn't correspond to an HTML id
			
			drawText ();
			
			e.stopPropagation();
			e.preventDefault();
		}
	}
}



function limit_3 (a, b, c)
{
    return a < b ? b : (a > c ? c : a);
}



function computeNextFloat (from, to, ease)
{
    return from + (to - from) * ease;
}



function toggleView (inDoItSlow)
{
	var midContDiv = document.getElementById ("Background-MiddleContainer");
	var midFillDiv = document.getElementById ("Background-MiddleFill");
	var rightContDiv = document.getElementById ("AllContent-RightSide-Container");
	
	var timeNow = (new Date).getTime();
	var multiplier = (inDoItSlow ? 10 : 1);							// enable slo-mo
	var startingSize = parseInt (midContDiv.clientWidth, 10);		// actual current size, even if already in motion right now
	var endingSize = gIsCollapsed ? kExpandedViewMiddleContainerWidth : kCollapsedViewMiddleContainerWidth;
	
	if (gIsCollapsed)												// going from collapsed to uncollapsed, so make the room to show it first
		window.resizeTo (kWidgetExpandedWidth, kWidgetHeight);		// if uncollapsed to collapsed, we will remove room after animation is done
	
	gResizeAnimation.elementMiddleContainer = midContDiv;			// for referral during animation
	gResizeAnimation.elementMiddleFill = midFillDiv;
	gResizeAnimation.elementRightContent = rightContDiv;
	
	if (gResizeAnimation.timer != null)								// it is moving already right now; interrupt the move and change to new size
	{
		clearInterval(gResizeAnimation.timer);
		gResizeAnimation.timer = null;
		gResizeAnimation.duration -= (timeNow - gResizeAnimation.startTime);
		gResizeAnimation.positionFrom = gResizeAnimation.positionNow;
	}
	else															// not moving already (normal case); arrange to start changing to new size
	{
		gResizeAnimation.duration = 250 * multiplier;
		gResizeAnimation.positionFrom = startingSize;
	}
	
	gResizeAnimation.positionTo = endingSize;
	gResizeAnimation.startTime = timeNow - 13;						// set it back one frame
	gResizeAnimation.onfinished = toggleViewAnimationFinished;
	
	gResizeAnimation.timer = setInterval ("toggleViewAnimate();", 13);
	toggleViewAnimate ();
	gIsCollapsed = !gIsCollapsed;
}



function toggleViewAnimate ()
{
	var T;
	var ease;
	var time  = (new Date).getTime();
	var xLoc;
	var frame;
	var rightContentClippingOnItsLeft;
	
	// Note: while the resizing of the widget is in motion due to this routine executing, mouseclicks
	// may cause dashboard to be dismissed.  It seems to be due to clicks coming into dashboard while
	// the elements' width are being changed.  This happens whether we set the element.style.width
	// directly or if we use element.style.setProperty().
	
	T = limit_3(time-gResizeAnimation.startTime, 0, gResizeAnimation.duration);
	ease = 0.5 - (0.5 * Math.cos(Math.PI * T / gResizeAnimation.duration));

	if (T >= gResizeAnimation.duration)
	{
		xLoc = gResizeAnimation.positionTo;
		clearInterval (gResizeAnimation.timer);
		gResizeAnimation.timer = null;
		
		if (gResizeAnimation.onfinished)
			setTimeout ("gResizeAnimation.onfinished();", 0); // call after the last frame is drawn
	}
	else
		xLoc = computeNextFloat(gResizeAnimation.positionFrom, gResizeAnimation.positionTo, ease);
	
	// Convert float to integer; old parseInt was like a round-down (floor), now Math.round rounds up:
	//
	gResizeAnimation.positionNow = Math.round(xLoc);
	
	// Affect the graphics to cause a frame of horizontal visual change: the main container that carries
	// with it the right endcap:
	//
	gResizeAnimation.elementMiddleContainer.style.width = gResizeAnimation.positionNow + "px";
	
	// The endcap is in the correct place before/during/after the move, but the background, whose width is
	// otherwise linked to that of the middle container, would be too wide and stick out beyond it (we don't
	// want to define its parent, the middle containter, to be smaller), so adjust it manually:
	//
	gResizeAnimation.elementMiddleFill.style.width = (gResizeAnimation.positionNow - 13) + "px";
	
	// Instead of a "wipe", we do a "push" of the right-hand-side content.  This can almost be achieved by
	// CSS settings as well:
	//
	gResizeAnimation.elementRightContent.style.left = (gResizeAnimation.positionNow - 145) + "px";
	
	// The "push" works correctly, but the right-hand-side content has alpha falloff shadows plus the left-hand-side
	// content has alpha gaps so that those mutually interfere.  So we clip the right-content-div on its
	// left edge as it slides under the center post:
	//
	rightContentClippingOnItsLeft = 296 - gResizeAnimation.positionNow;
	gResizeAnimation.elementRightContent.style.clip = "rect(0,999,999," + rightContentClippingOnItsLeft + ")";
}



function toggleViewAnimationFinished ()
{
	// Set the overall widget's window size so that screen captures, mouseover for the close box, etc, work
	// correctly for the resized widget (for example, don't reference empty space where the expanded view was).
	// We do this when the animation is done, rather than during, for smoothness reasons.  If we expanded then
	// this was already done; if we collapsed then reduce the window size now:
	//
	if (gIsCollapsed)
		window.resizeTo (kWidgetCollapsedWidth, kWidgetHeight);
	
	// After a collapse/expand operation we always want to snap back to the current date:
	//
	jumpToToday ();
	drawText ();
	if (window.widget)
		widget.setPreferenceForKey (gIsCollapsed ? "true" : "false", createKey("collapsed"));
}



function idFromMonthDate (month, dateCount)
{
	return ((month<10) ? "0" + month : month.toString()) + 
	       ((dateCount<10) ? "0" + dateCount : dateCount.toString());
}



function setupWeekHeader ()
//
// Get localized names of the narrow days of the week from the plugin and populate the Expanded View's calendar-grid's header:
//
{
	for (var i = 0;   i <= 6;   i++)
	{
		document.getElementById ("DayOfWeek-" + i).innerText = fetchIPrefNarrowDayOfWeekName (i);
	}
}



function myGetDayFromDate (inDate)
//
// JavaScript's Date.getDay() is Sunday-based, so we have to fix it up.
//
{
	var dayIndex = inDate.getDay() - fetchIPrefFirstWeekday();
	if (dayIndex < 0)
		dayIndex += 7;
	return dayIndex;
}



function calculateNumberOfDaysInPreviousMonth (inYear, inMonth)
//
// Return the number of days in the month that is previous to the one given, going into the previous year if needed.
//
{
	var returnValue;
	var monthOfInterestYear, monthOfInterestMonth;
	var aTrialDate;
	
	if (inMonth != 0)
	{
		monthOfInterestYear = inYear;
		monthOfInterestMonth = inMonth - 1;
	}
	else
	{
		monthOfInterestYear = inYear - 1;
		monthOfInterestMonth = 11;
	}
	
	aTrialDate = new Date (monthOfInterestYear, monthOfInterestMonth, 1);
	for (var i = 27;   i < 32;   i++)
	{
		aTrialDate.setDate (i);
		if (aTrialDate.getMonth () != monthOfInterestMonth)
			break;
		else
			returnValue = i;
	}
	
	return returnValue;
}



function drawGrid (inMonth, inYear)
//
// Draw the calendar day-names-and-date-numbers grid.
//
{
	var i;
	
	// we will fill this DIV in
	var monthDiv = document.getElementById ("Calendar-NumberGrid");
	
	// show the names of the days of the week
	setupWeekHeader ();
	
	// clear out any old calendar grid numbers out of the DIV
	if (monthDiv.hasChildNodes ())
	{
		var children = monthDiv.childNodes;
		var count = children.length;
		for (i = 0;   i < count;   i++)
		{
			monthDiv.removeChild (children[0]);
		}
	}
	
	// create the month grid, filling the DIV
	var dateNumberThatIsValidInOurMonth = 1;
	var dateToCheckDayRangeInOurMonth = new Date (inYear, inMonth, dateNumberThatIsValidInOurMonth);
	var zeroBasedFirstDayOfWeekInOurMonth = myGetDayFromDate (dateToCheckDayRangeInOurMonth);
	var numberOfDaysInPreviousMonth = calculateNumberOfDaysInPreviousMonth (inYear, inMonth);
	var dateNumberThatIsValidInNextMonth = 1;
	for (i = 0;   i < 42;   i++)
	{
		var dateSpan = document.createElement ("span");
		
		if (i < zeroBasedFirstDayOfWeekInOurMonth)
		{
			// show dimmed numbers of the (final days of the previous month) before the begin of our month
			dateSpan.innerHTML = (numberOfDaysInPreviousMonth - zeroBasedFirstDayOfWeekInOurMonth) + i + 1;
			dateSpan.setAttribute ("class", "calendar-dayOfMonthNumberDimmed");
			
			// increment: we are waiting for the main loop index (i) to get into range of our month, so do nothing else
		}
		else
		{
			if (dateToCheckDayRangeInOurMonth.getMonth () != inMonth)
			{
				// show dimmed numbers of the (first days of the next month) after the end of our month
				dateSpan.innerHTML = dateNumberThatIsValidInNextMonth;
				dateSpan.setAttribute ("class", "calendar-dayOfMonthNumberDimmed");
				
				// increment: only the simple 1,2,3... number of next month now that we are in the beyond-state
				dateNumberThatIsValidInNextMonth++;
			}
			else
			{
				// show normal numbers of the days in our month
				// they are clickable: they trigger event handlers when clicked and have
				// a formatted setting for the ID attribute that is used by the even handler to identify the date
				dateSpan.innerHTML = dateNumberThatIsValidInOurMonth;
				dateSpan.setAttribute ("class", "calendar-dayOfMonthNumberNormal");
				dateSpan.setAttribute ("id", idFromMonthDate (inMonth, dateNumberThatIsValidInOurMonth));
				dateSpan.setAttribute("onmousedown", "selectDate(event, this.id);");
				dateSpan.setAttribute("onkeydown", "selectDate(event, this.id);");
				
				// increment: bump the valid date number in our month and bump the special date we are using to see
				// when our main loop goes beyond the number of days in our month
				dateNumberThatIsValidInOurMonth++;
				dateToCheckDayRangeInOurMonth.setDate (dateNumberThatIsValidInOurMonth);
			}
		}
		
		monthDiv.appendChild (dateSpan);
	}
}



function changeMonth (inEvent, inMoveNext)
{
	if (inMoveNext)
	{
		gBrowsingMonth++;
		if (gBrowsingMonth > 11)
		{
			gBrowsingMonth = 0;
			gBrowsingYear++;
		}
	}
	else
	{
		gBrowsingMonth--;
		if (gBrowsingMonth < 0)
		{
			gBrowsingMonth = 11;
			gBrowsingYear--;
			if (gBrowsingYear < 1753)		// pin backwards monthwise browsing at January 1753; see note above
			{
				gBrowsingMonth = 0;
				gBrowsingYear = 1753;
			}
		}
	}
	
	// create the new calendar
	drawGrid (gBrowsingMonth, gBrowsingYear);
	
	// set date selections as necessary
	drawHiliteTodayInGrid ();
		
	gBrowsingDate = 0; // clear the selected date
	
	drawText ();
}



function drawHiliteTodayInGrid ()
//
// If the current calendar-grid being shown is of the now-month-and-year, highlight the corresponding date-number in it.
// This shows to the user that today's date is in there.  Otherwise, don't, ie, if the user is browsing into
// other months.
//
{
	if ((gBrowsingYear == gDate.getFullYear ()) &&  (gBrowsingMonth == gDate.getMonth ()))
	{
		var todayCalendarNumbersGridElement = document.getElementById (idFromMonthDate (gBrowsingMonth, gDate.getDate()));
		todayCalendarNumbersGridElement.setAttribute ("class", "calendar-dayOfMonthNumberHighlighted");
	}
}



function jumpToToday (inEvent)
{
	gDate = new Date ();
	gBrowsingMonth = gDate.getMonth ();
	gBrowsingYear = gDate.getFullYear ();
	drawGrid (gBrowsingMonth, gBrowsingYear);
	drawHiliteTodayInGrid ();
	gBrowsingDate = gDate.getDate ();
	drawText ();
}



function selectDate (inEvent, inID)
//
// Can be used to take action when a date is clicked upon.
//
{
}



function fetchIPrefFirstWeekday () // returns integer
//
// Fetch data from this widget's plugin, from International Prefs.  Do fallback work if no plugin available.
//
// Get the first-day-of-week, 0-based index.
//
{
	if (window.Calendar)				// plugin available, and if plugin error will return 0
	{
		return Calendar.firstWeekday ();
	}
	else								// fallback case for if no plugin or plugin error
	{
		return 0;
	}
}



function fetchIPrefNarrowDayOfWeekName (inDayOfWeekIndex) // returns string
//
// Fetch data from this widget's plugin, from International Prefs.  Do fallback work if no plugin available.
//
// Get the (localized customized in International Pref Pane) "narrow" day-of-week, which is usually a single
// character such as "S" for Sunday, etc.  The inDayOfWeek input is zero-based and follows USA first-day-of-week
// conventions, even if the first-day-of-week is not Sunday for the current localized settings.  Thus: 0=Sunday,
// 1=Monday, 2=Tuesday, 3=Wednesday, 4=Thursday, 5=Friday, and 6=Saturday, in a single-character format.
//
{
	var returnString = undefined;
	
	if (window.Calendar)				// plugin available: attempt to fill the string; will fill with undefined if plugin error
	{
		returnString = Calendar.localizedNarrowDayOfWeekNameForIndex (inDayOfWeekIndex);
	}
	
	if (returnString == undefined)		// still undefined: fallback if plugin unvailable or plugin returned null (plugin error)
	{
		switch (inDayOfWeekIndex)
		{
			case 0:   returnString = "S";   break;
			case 1:   returnString = "M";   break;
			case 2:   returnString = "T";   break;
			case 3:   returnString = "W";   break;
			case 4:   returnString = "T";   break;
			case 5:   returnString = "F";   break;
			case 6:   returnString = "S";   break;
		}
	}
	
	return returnString;
}



function fetchIPrefLongMixedcaseDayOfWeekName (inDayOfWeekIndex) // returns string
//
// Fetch data from this widget's plugin, from International Prefs.  Do fallback work if no plugin available.
//
// Get the (localized customized in International Pref Pane) mixed-case long day name,
// zero-based.  For example, 0="Sunday", 1="Monday", etc.
//
{
	var returnString = undefined;
	
	if (window.Calendar)				// plugin available: attempt to fill the string; will fill with undefined if plugin error
	{
		returnString = Calendar.localizedWideDayOfWeekNameForIndex (inDayOfWeekIndex);
	}
	
	if (returnString == undefined)		// still undefined: fallback if plugin unvailable or plugin returned null (plugin error)
	{
		switch (inDayOfWeekIndex)
		{
			case 0:   returnString = "Sunday";		break;
			case 1:   returnString = "Monday";		break;
			case 2:   returnString = "Tuesday";		break;
			case 3:   returnString = "Wednesday";	break;
			case 4:   returnString = "Thursday";	break;
			case 5:   returnString = "Friday";		break;
			case 6:   returnString = "Saturday";	break;
		}
	}
	
	return returnString;
}



function fetchIPrefLongMonthNameAndYearNumber (inYear, inMonthIndex) // returns string
//
// Fetch data from this widget's plugin, from International Prefs.  Do fallback work if no plugin available.
//
// Get the (localized customized in International Pref Pane) combined string of the form "January 1900", that is,
// the long mixed-case month name followed by a 4-digit year number (for most locales) and a reversed or otherwise
// modified version of the same (for other locales).
//
{
	var returnString = undefined;
	
	if (window.Calendar)				// plugin available: attempt to fill the string; will fill with undefined if plugin error
	{
		returnString = Calendar.localizedWideMonthNameAndYearNumberForYearNumberAndMonthIndex (inYear, inMonthIndex);
	}
	
	if (returnString == undefined)		// still undefined: fallback if plugin unvailable or plugin returned null (plugin error)
	{
		switch (inMonthIndex)
		{
			case 0:    returnString = "January";		break;
			case 1:    returnString = "February";		break;
			case 2:    returnString = "March";			break;
			case 3:    returnString = "April";			break;
			case 4:    returnString = "May";			break;
			case 5:    returnString = "June";			break;
			case 6:    returnString = "July";			break;
			case 7:    returnString = "August";			break;
			case 8:    returnString = "September";		break;
			case 9:    returnString = "October";		break;
			case 10:   returnString = "November";		break;
			case 11:   returnString = "December";		break;
		}
		returnString = returnString + " " + inYear.toString ();
	}
	
	return returnString;
}



function setFontSizeToFitAndCenter (inElement, inStartingFontSize, inLeft, inStartingTop, inMaxWidth, inMaxHeight)
//
// Sets the font size of the given element, while trying to ensure that the string contained within that
// element fits without clipping or spilling in the given width/height pixel size.
//
// The starting font size and top position parameters are given by the caller so that the setting always
// proceeds from an absolute basis.  If the requested font size cannot be honored, then the font size and
// top position in the element are repeatedly adjusted until the element's inner text fits (up to a limit).
//
// NOTE: The CSS declaration for the given element must not have explicitly specified a "width" property.
// NOTE: The height parameter is useful even for "single line" cases to prevent wrapping to an undesired
// additional line.
//
{
	var measuredWidth;
	var measuredHeight;
	var newFontSize = inStartingFontSize;
	var newTop = inStartingTop;
	var numReductions = 0;
	
	while (1)
	{
		inElement.style.setProperty ("font-size", newFontSize + "px", "important");
		inElement.style.setProperty ("top", newTop + "px", "important");
		measuredWidth = parseInt (document.defaultView.getComputedStyle (inElement, null).getPropertyValue ("width"), 10);
		measuredHeight = parseInt (document.defaultView.getComputedStyle (inElement, null).getPropertyValue ("height"), 10);
		inElement.style.setProperty ("left", inLeft + (inMaxWidth / 2) - (measuredWidth / 2) + "px", "important");
		if ((measuredWidth > inMaxWidth) || (measuredHeight > inMaxHeight))
		{
			if (newFontSize <= 6)
			{
				// this is the limit of of how small we will make it:
				break;
			}
			// adjust the size, then loop around one more time to set it; after a
			// few iterations, slow down the vertical adjustment a bit to better
			// track smaller font sizes' change; ideally this would be measured:
			numReductions++;
			newFontSize -= 1;
			if (numReductions <= 4) newTop += 1; else newTop += 0.8;
		}
		else
		{
			// this is where we exit if the size is acceptable:
			break;
		}
	}
}



function setFontSizeToFitAndCenterFix (inElement, inStartingFontSize, inLeft, inStartingTop, inMaxWidth, inMaxHeight)
//
// This is a workaround for a possible quirk with text measurement.  It is desirable to fix the
// underlying problem rather than rely on the loop below, because the change causes a visible size
// change in the text size, yet keeping it hidden and having a setTimeout proc to enable visibility
// clutters our code and causes flashing while browsing.  The problem is that in setFontSizeToFit, the
// getComputedStyle (inElement, null).getPropertyValue ("width") returns a different (smaller) value
// the first time than subsequent times.  This has to do with when things are rendered async; maybe
// set a proc to be called upon render completion?
//
{
	for (var i = 1;   i <= 3;   i++)
	{
		setFontSizeToFitAndCenter (inElement, inStartingFontSize, inLeft, inStartingTop, inMaxWidth, inMaxHeight);
	}
}



function drawText ()
//
// Updates the non-calendar-grid text strings such as the (day-name), (day-of-month-number), and (monthname+yearnumber).
//
{
	var dayOfWeekNameElement			= document.getElementById ("Message-DayOfWeekName");
	var dateOfMonthNumberElement		= document.getElementById ("Message-DateOfMonthNumber");
	var monthNameAndYearNumberElement	= document.getElementById ("Message-MonthNameAndYearNumber");
	var dateNumber;
	
	// This is, ie, "Wednesday" (the day-of-week name) on the left side; it always reflects the current time.
	// We do shrink-to-fit if needed: in some locales, this string can grow quite long:
	//
	dayOfWeekNameElement.innerText = fetchIPrefLongMixedcaseDayOfWeekName (myGetDayFromDate (gDate));
	setFontSizeToFitAndCenterFix (dayOfWeekNameElement,
						/* starting font size: */		18,
						/* starting left position: */	7,
						/* starting top position: */	9,
						/* max width: */				134,
						/* max height: */				25);
	
	// This is, ie, "31" (the big date number) on the left side; it always reflects the current time.
	// We dynamically adjust details of position and kerning to get the big huge number to look right:
	//
	dateNumber = gDate.getDate ();
	dateOfMonthNumberElement.innerText = dateNumber;
	if (dateNumber <= 9) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style1to9");
	else
	if (dateNumber == 10) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style10to10");
	else
	if (dateNumber == 11) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style11to11");
	else
	if ((dateNumber >= 12) && (dateNumber <= 19)) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style12to19");
	else
	if (dateNumber == 20) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style20to20");
	else
	if (dateNumber == 21) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style21to21");
	else
	if ((dateNumber >= 22) && (dateNumber <= 26)) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style22to26");
	else
	if ( dateNumber == 27) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style27to27");
	else
	if ((dateNumber >= 28) && (dateNumber <= 29)) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style28to29");
	else
	if (dateNumber == 30) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style30to30");
	else
	if (dateNumber == 31) dateOfMonthNumberElement.setAttribute ("class", "message-dateOfMonthNumber-style31to31");
	
	// This is, ie, "January 1900" (the month + year combination) on the right side; it reflects what is currently being browsed.
	// We do shrink-to-fit if needed: in some locales, this string can grow quite long:
	//
	monthNameAndYearNumberElement.innerText = fetchIPrefLongMonthNameAndYearNumber (gBrowsingYear, gBrowsingMonth);
	setFontSizeToFitAndCenterFix (monthNameAndYearNumberElement,
						/* starting font size: */		12,
						/* starting left position: */	-5,
						/* starting top position: */	9,
						/* max width: */				120,
						/* max height: */				25);
}



function mouseDownForButton (inEvent, inThis)
{
	inEvent.target.src = "Images/"+inThis.myName+"_pressed.png";
	inEvent.stopPropagation();
	inEvent.preventDefault();
	document.addEventListener ("mouseup", mouseUpForButtonsInsideOutside, true);
	gButtonBeingTracked = inThis;
}



function mouseUpForButtonsInsideOutside (inEvent)
{
	// we use this event listener for the global document instead of an event handler on the button <IMG> so
	// that we can know to stop the tracking process even if the mouseup happens outside of the button
	
	if (null != gButtonBeingTracked)
		gButtonBeingTracked.src = "Images/"+gButtonBeingTracked.myName+".png";
	
	// note that inEvent.target will point to the IMG object only if this is a mouseup inside the button,
	// and also make sure we don't respond to mouseup in another button, hence the extra check for this
	// event being inside the button being tracked
	
	if (inEvent.target == gButtonBeingTracked)
	{
		if (inEvent.target.myName == "bar_left")
		{
			changeMonth(inEvent, false);
		}
		else if (inEvent.target.myName == "bar_right")
		{
			changeMonth(inEvent, true);
		}
	}
	
	inEvent.stopPropagation();
	inEvent.preventDefault();
	
	document.removeEventListener ("mouseup", mouseUpForButtonsInsideOutside, true);
	gButtonBeingTracked = null;
}



function mouseOutForButton (inEvent, inThis)
{
	inEvent.target.src = "Images/"+inThis.myName+".png";
}



function mouseOverForButton (inEvent, inThis)
{
	if (inThis == gButtonBeingTracked)
		inEvent.target.src = "Images/"+inThis.myName+"_pressed.png";
}



function onRequestClickOnLeftSide (inEvent)
{
	toggleView (inEvent.shiftKey);
}



function onRequestClickedMonthNameAndYearNumber (inEvent)
{
	jumpToToday (inEvent);
}



function setupUpdateTimer ()
{
	if (gUpdateDateTimer != null)
	{
		clearTimeout (gUpdateDateTimer);
		gUpdateDateTimer = null;
	}
		
	// create a new timer for when we need to update the date
	var hourMillis = ((23 - gDate.getHours()) * 3600000);
	var minsMillis = ((59 - gDate.getMinutes()) * 60000);
	var secsMillis = ((60 - gDate.getSeconds()) * 1000); // overshoot a second
	var millisRemaining = hourMillis + minsMillis + secsMillis;
	
	gUpdateDateTimer = setTimeout("updateDate();", millisRemaining);
}



function updateDate ()
{
	var currentEqualDate = gDate.getMonth() == gBrowsingMonth &&
						  gDate.getFullYear() == gBrowsingYear &&
						  gDate.getDate() == gBrowsingDate;
	gDate = new Date();
	
	if (currentEqualDate)
	{
		// only change the current vars if they matched the date
		// before it was updated
		gBrowsingMonth = gDate.getMonth();
		gBrowsingYear = gDate.getFullYear();
		gBrowsingDate = gDate.getDate();
	}
	
	drawGrid (gBrowsingMonth, gBrowsingYear); 
	drawHiliteTodayInGrid ();
	drawText ();
	
	// reset the timer
	setupUpdateTimer();
}



function onPluginNotify ()
{
	// refresh the date, because the plugin is notifying us that date and/or format changed
	
	// note that onshow might also do this so multiple updates could happen but that is OK for now; having
	// the notify come from the plugin covers cases where the date is changed on us while the dashboard is
	// active and frontmost (remote login or another widget changing the date); further, it causes the date
	// to not "jump" from old to new if it WAS a case of the date changing while dashboard was frontmost
	
	updateDate ();
}



function onshow ()
{
	// refresh the date, in case it changed while the dashboard was inactive or hidden
	
	// note that onPluginNotify might also do this so multiple updates can happen but that is OK for now;
	// having the notify be handled here besides just relying on the plugin is good because the plugin
	// might not be installed
	
	updateDate ();
}



function onhide ()
{
	if (gUpdateDateTimer)
	{
		clearTimeout(gUpdateDateTimer);
		gUpdateDateTimer = null;
	}
}



function onLoaded ()
{
	// Obtain pointers to some HTML elements:
	//
	var midContDiv = document.getElementById ("Background-MiddleContainer");
	var midFillDiv = document.getElementById ("Background-MiddleFill");
	var rightContDiv = document.getElementById ("AllContent-RightSide-Container");
	var leftButtonImg = document.getElementById ("Button-Left");
	var rightButtonImg = document.getElementById ("Button-Right");
	
	// Add some ALT tags so that an alternative view of the widget can work: for example, VoiceOver
	// can speak the names of items like button images:
	//
	leftButtonImg.setAttribute ("alt", getLocalizedString ("Previous Month Button"));
	rightButtonImg.setAttribute ("alt", getLocalizedString ("Next Month Button"));
	
	// Register for notifications:
	//
	if (window.widget)
	{
		widget.onhide = onhide;
		widget.onshow = onshow;
	}
	
	// Obtain persistent pref settings; default to Expanded View if no pref defined:
	//
	var viewSize = "expanded";
	if (window.widget)
	{
		var pref = widget.preferenceForKey (createKey("collapsed"));
		if (pref != undefined)
		{
			if (pref == "true")
			{
				viewSize = "collapsed";
			}
		}
	}
	
	// Initialize the global "gIsCollapsed", the middle container div size, the middle container div's background fill size,
	// and the total window size on startup:
	//
	switch (viewSize)
	{
	case "collapsed":
		gIsCollapsed = true;
		midContDiv.style.width = kCollapsedViewMiddleContainerWidth + "px";
		midFillDiv.style.width = kCollapsedViewMiddleFillWidth + "px";
		rightContDiv.style.left = kCollapsedViewRightContentLeftPosition + "px";
	break;
	
	case "expanded":
		gIsCollapsed = false;
		midContDiv.style.width = kExpandedViewMiddleContainerWidth + "px";
		midFillDiv.style.width = kExpandedViewMiddleFillWidth + "px";
		rightContDiv.style.left = kExpandedViewRightContentLeftPosition + "px";
	break;
	}
	
	// Draw things:
	//
	drawGrid (gBrowsingMonth, gBrowsingYear);
	drawText ();
	drawHiliteTodayInGrid ();
	
	// Register our keypress handler:
	//
	document.addEventListener ("keypress", keyPressed, true);
}



function onremove ()
{
	widget.setPreferenceForKey (null, createKey("collapsed"));
}



function createKey (key)
{
	return widget.identifier + "-" + key;
}



function getLocalizedString (inKey)
{
	try
	{
		var outString = localizedStrings [inKey];
		if (outString == undefined) return inKey;
		return outString;
	}
	catch (ex)
	{}
	
	return inKey;
}



if (window.widget)
{
	widget.onremove = onremove;
}



